<?php
/**
 * @version    3.0.2
 * @package    com_gagallery
 * @author     Glenn Arkell <glenn@glennarkell.com.au>
 * @copyright  2021 Glenn Arkell
 * @license    GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace GlennArkell\Component\Gagallery\Administrator\Helper;
// No direct access
defined('_JEXEC') or die;

use \Joomla\CMS\Factory;
use \Joomla\CMS\Language\Text;
use \Joomla\CMS\User\UserHelper;
use \Joomla\CMS\User\UserFactoryInterface;
use \Joomla\CMS\Uri\Uri;
use \Joomla\CMS\Component\ComponentHelper;
use \Joomla\Filesystem\Path;
use \Joomla\Filesystem\File;
use \Joomla\Filesystem\Folder;
use \Joomla\CMS\User\User;
use \Joomla\CMS\Date\Date;
use \Joomla\CMS\Access\Access;
use \Joomla\CMS\Installer\Installer;
use \Joomla\Session\SessionInterface;
use \Joomla\CMS\Application\SiteApplication;

/**
 * Gagallery helper.
 * @since  1.6
 */
class GagalleryHelper
{
	/**
	 * Build the HTTP query array
	 * @param array of the http query if the http query already prepared and we want to add more
	 * @param string view or a task
	 * @param string control method to be used
	 * @param string reference indicator
	 * @param string/int reference string or id
	 * @return array
	 * @ hint - this is used with http_build_query($query_string, '', '&amp;') to build a clean url
	 */
	public static function getHTTPQuery($existQ = null, $viewTask = 'view', $contModel = 'picture', $ref = 'id', $linkId = 0)
	{
        if ($contModel == 'article' || $contModel == 'category') {
            $compName = 'com_content';
        } else {
            $compName = 'com_gagallery';
        }
        if (!$existQ) {
			$query_string = array();
			$query_string['option'] = $compName;
			$query_string[$viewTask] = $contModel;
			if ($ref) {
				$query_string[$ref] = $linkId;
			}
		} else {
			$query_string = $existQ;
			$query_string[$ref] = $linkId;
		}

		return $query_string;
	}

    /**
     * Gets todays date object based on global timezone settings
     */
    public static function getTodaysDate()
	{
		$tz = Factory::getConfig()->get('offset');
		$date = Factory::getDate('now', $tz);
		//$today = date_format($date,'Y-m-d H:i:s');

		return $date;
	}

    /**
     * Gets the user record for the specific id reference
     */
    public static function getSpecificUser($id = 0)
	{
		if ($id) {
			$container = Factory::getContainer();
			$userFactory = $container->get(UserFactoryInterface::class);
			$user = $userFactory->loadUserById($id);
		} else {
			$user = Factory::getApplication()->getIdentity();
		}
		unset($user->password);
		unset($user->activation);
		unset($user->params);
		unset($user->lastResetTime);
		unset($user->resetCount);
		unset($user->otpKey);
		unset($user->otep);
		unset($user->requireReset);

		return $user;
	}

	/**
	 * Set up a web asset object for use
	 */
	public static function setupGAWebApp()
	{
        // Boot the DI container.
        $container = Factory::getContainer();
        // Alias the session service key to the web session service.
        $container->alias(SessionInterface::class, 'session.web.site');
        // Get the application.
        $app      = $container->get(SiteApplication::class);
        //$template = $app->getTemplate(true);
        //$params   = $template->params;
        //$wa = Factory::getApplication()->getDocument()->getWebAssetManager();
        
        return $app;
	}

    /**
     * Load template over-rides when using modal view
     * This is important for Cloud White template scheme
     * because modals have the primary css over-ride any
     * css loaded in the template index.php file
     */
    public static function loadTmplStyleModal($wa)
	{
        $tmpl = Factory::getApplication()->getTemplate(true);
        if ($tmpl->params->get('colorName') == 'colors_white') {
        	$wa->addInlineStyle('.btn-primary {background-color:'.$tmpl->params->get('btnPbgColor').' !important;}');
        	$wa->addInlineStyle('.nav-link {color:var(--template-contrast) !important;}');
        	$wa->addInlineStyle('.form-check-input:checked, .form-select[multiple] option:checked, [multiple].custom-select option:checked {background-color:'.$tmpl->params->get('btnPbgColor').' !important;}');
        	$wa->addInlineStyle('.page-item.active .page-link {color: var(--rkic41site-color-link); background-color: #c1cee1 !important; border-color: #dfe3e7 !important;}');
        	$wa->addInlineStyle('.form-select[multiple] option:checked, [multiple].custom-select option:checked {background-color: var(--rkic41site-color-primary-border) !important;}');
        }
		return true;
	}

	/**
	 * Get the version of the component
	 * @return version element of manifest
	 */
	public static function getComponentVersion()
	{
		$componentXML = Installer::parseXMLInstallFile(Path::clean(JPATH_ADMINISTRATOR . '/components/com_gagallery/gagallery.xml'));

		return $componentXML['version'];
	}

	/**
	 * Gets the category object of an item
	 * @param   int     $id     The item's id
	 * @return  object  record
	 */
	public static function getCategory($id)
	{
		$db = Factory::getContainer()->get('DatabaseDriver');
		$query = $db->getQuery(true);

		$query
			->select('*')
			->from('#__categories')
			->where('id = ' . (int) $id);

		$db->setQuery($query);

		return $db->loadObject();
	}

    /**
     * Gets the edit permission for an user
     * @param   mixed  $item  The item
     * @return  bool
     */
    public static function canUserEdit($user, $item)
    {
        $permission = false;

        if ($user->authorise('core.edit', 'com_gagallery')) {
            $permission = true;
        } else {
            if (isset($item->created_by) && $item->created_by) {
                if ($user->authorise('core.edit.own', 'com_gagallery') && $item->created_by == $user->id) {
                    $permission = true;
                }
            } else {
                if ($user->authorise('core.create', 'com_gagallery')) {
                    $permission = true;
                }
            }
        }

        return $permission;
    }

	/**
	 * Get the record
	 * @params integer Record id reference
	 * @return object list of records
	 */
	public static function getArticle($id = 0)
	{
        $db		= Factory::getContainer()->get('DatabaseDriver');
		$query	= $db->getQuery(true);
        $query->clear();
		$query->select(' a.* ');
		$query->from(' #__content AS a ');
		$query->where('a.id = '.(int) $id);
		$db->setQuery((string)$query);
	    try {
	        return $db->loadObject();
	    } catch (\RuntimeException $e) {
	        Factory::getApplication()->enqueueMessage($e->getMessage(), 'danger');
	        return false;
	    }
	}

	public static function getListOptions($table = '#__gagallery_pictures', $label = 'Picture', $fld_value = 'id', $fld_text = 'title' )
	{
        // setup for other languages regarding the selection
        $sel_label = strtoupper('COM_GAGALLERY_SELECT_'.$label);
        $sel_label = Text::_($sel_label);
		// get the records
        $db = Factory::getContainer()->get('DatabaseDriver');
		$query	= $db->getQuery(true);
        $query->clear();
		$query->select(' "0" as "value", "'.$sel_label.'" as "text" UNION SELECT '.$fld_value.' as value, '.$fld_text.' as text ');
		$query->from( $table );
		if ($table == '#__users') {
            $query->where(' block = 0' );
		} else if ($table == '#__hikashop_category') {
            $query->where(' category_type = '. $db->Quote('product') );
            $query->where(' category_name != '. $db->Quote('product category') );
            $query->where(' category_published = 1' );
        } else {
            $query->where(' state = 1' );
        }
		$query->order(' text ASC ' );
		$db->setQuery((string)$query);

	    try {
	        $options = $db->loadObjectList();
	    } catch (RuntimeException $e) {
	        Factory::getApplication()->enqueueMessage($e->getMessage(),'danger');
	        return false;
	    }

		return $options;
	}

	public static function getCategoryOptions($ext = 'com_gagallery', $label = 'Category Type', $field = 'title' )
	{
		// get the categories
        $db = Factory::getContainer()->get('DatabaseDriver');
		$query	= $db->getQuery(true);
        $query->clear();
		$query->select(' "0" as "value", " - Select '.$label.' - " as "text" UNION SELECT id as value, '.$field.' as text ');
		$query->from( ' #__categories ' );
		$query->where(' published = 1' );
		$query->where(' extension = '.$db->quote($ext) );
		$query->order(' text ASC ' );
		$db->setQuery((string)$query);

	    try {
	        $options = $db->loadObjectList();
	    } catch (RuntimeException $e) {
	        Factory::getApplication()->enqueueMessage($e->getMessage(),'danger');
	        return false;
	    }

		return $options;
	}

    /**
    *   Method to get values for a member based on field id
    *   @param int $id field_id reference
    *   @param int $userId user id reference
    *   @return object field name and value
    */
	public static function getCustomFieldValue($id = 0, $userId = 0)
	{
        $db		= Factory::getContainer()->get('DatabaseDriver');
		$query	= $db->getQuery(true);
        $query->clear();
		$query->select(' b.title As fieldName, a.value AS fieldValue ');
		$query->from(' #__fields_values AS a ');
		$query->join('LEFT', ' #__fields AS b ON b.id = a.field_id');
		$query->where(' a.field_id = '.(int) $id );
		$query->where(' a.item_id = '.(int) $userId );
		$db->setQuery((string)$query);

	    try {
	        return $db->loadObject();
	    } catch (RuntimeException $e) {
	        Factory::getApplication()->enqueueMessage($e->getMessage().' Field Values');
	        return false;
	    }

	}

    /**
    *   Method to get the required record
    *   @param string $table table name
    *   @param int $id id reference
    *   @return object record data
    */
	public static function getRecord($table = '', $id = 0)
	{
        $db		= Factory::getContainer()->get('DatabaseDriver');
		$query	= $db->getQuery(true);
        $query->clear();
		$query->select(' a.* ');
		$query->from($db->quoteName($table, 'a'));
		$query->where(' a.id = '.(int) $id );
		$db->setQuery((string)$query);

	    try {
	        return $db->loadObject();
	    } catch (RuntimeException $e) {
	        Factory::getApplication()->enqueueMessage($e->getMessage().' Record');
	        return false;
	    }

	}

    /**
    *   Method to get the required record
    *   @param string $table table name
    *   @param string $field field name to match on the id reference
    *   @param int $id id reference
    *   @return object record data
    */
	public static function getRecordList($table = '', $field = '', $id = 0)
	{
        $db		= Factory::getContainer()->get('DatabaseDriver');
		$query	= $db->getQuery(true);
        $query->clear();
		$query->select(' a.* ');
		$query->from($db->quoteName($table, 'a'));
		$query->where($db->quoteName($field).' = '.(int) $id );
		$db->setQuery((string)$query);

	    try {
	        return $db->loadObjectList();
	    } catch (RuntimeException $e) {
	        Factory::getApplication()->enqueueMessage($e->getMessage().' RecordList');
	        return false;
	    }

	}

}

