<?php
/**
 * @version    3.0.2
 * @package    com_gagallery
 * @author     Glenn Arkell <glenn@glennarkell.com.au>
 * @copyright  2021 Glenn Arkell
 * @license    GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace GlennArkell\Component\Gagallery\Site\Model;

// No direct access.
defined('_JEXEC') or die;

use \Joomla\CMS\Factory;
use \Joomla\Utilities\ArrayHelper;
use \Joomla\CMS\Language\Text;
use \Joomla\CMS\Table\Table;
use \Joomla\CMS\MVC\Model\FormModel;
use \Joomla\Filesystem\Path;
use \Joomla\Filesystem\File;
use \Joomla\Filesystem\Folder;
use \Joomla\CMS\Component\ComponentHelper;
use \GlennArkell\Component\Gagallery\Administrator\Helper\GagalleryHelper;

/**
 * Form model.
 * @since  1.6
 */
class PictureformModel extends FormModel
{
    private $item = null;

    /**
     * Method to auto-populate the model state.
     * Note. Calling getState in this method will result in recursion.
     * @return void
     * @since  1.6
     * @throws Exception
     */
    protected function populateState()
    {
        $app = Factory::getApplication('com_gagallery');

        // Load state from the request userState on edit or from the passed variable on default
        if ($app->input->get('layout') == 'edit') {
                $id = $app->getUserState('com_gagallery.edit.picture.id');
        } else {
                $id = $app->input->get('id');
                $app->setUserState('com_gagallery.edit.picture.id', $id);
        }

        $this->setState('picture.id', $id);

        // Load the parameters.
        $params       = $app->getParams();
        $params_array = $params->toArray();

        if (isset($params_array['item_id'])) {
                $this->setState('picture.id', $params_array['item_id']);
        }

        $this->setState('params', $params);
    }

    /**
     * Method to get an ojbect.
     * @param   integer $id The id of the object to get.
     * @return Object|boolean Object on success, false on failure.
     * @throws Exception
     */
    public function getItem($id = null)
    {
        if ($this->item === null) {
            $this->item = false;

            if (empty($id)) {
                    $id = $this->getState('picture.id');
            }

            // Get a level row instance.
            $table = $this->getTable();

            if ($table !== false && $table->load($id) && !empty($table->id)) {
                $user = GagalleryHelper::getSpecificUser();
                $id   = $table->id;

                $canEdit = $user->authorise('core.edit', 'com_gagallery') || $user->authorise('core.create', 'com_gagallery');

                if (!$canEdit && $user->authorise('core.edit.own', 'com_gagallery')) {
                    $canEdit = $user->id == $table->created_by;
                }

                if (!$canEdit) {
                    throw new \Exception(Text::_('JERROR_ALERTNOAUTHOR'), 403);
                }

                // Check published state.
                if ($published = $this->getState('filter.published')) {
                    if (isset($table->state) && $table->state != $published) {
                        return $this->item;
                    }
                }

                // Convert the Table to a clean Object.
                $properties = $table->getProperties(1);
                $this->item = ArrayHelper::toObject($properties, 'stdClass');

            }
        }

        return $this->item;
    }

    /**
     * Method to get the table
     * @param   string $type   Name of the Table class
     * @param   string $prefix Optional prefix for the table class name
     * @param   array  $config Optional configuration array for Table object
     * @return  Table|boolean Table if found, boolean false on failure
     */
    public function getTable($type = 'Picture', $prefix = 'Administrator', $config = array())
    {
        return parent::getTable($type, $prefix, $config);
    }

    /**
     * Get an item by alias
     * @param   string $alias Alias string
     * @return int Element id
     */
    public function getItemIdByAlias($alias)
    {
        $table      = $this->getTable();
        $properties = $table->getProperties();

        if (!in_array('alias', $properties)) {
                return null;
        }

        $table->load(array('alias' => $alias));
        $id = $table->id;

        return $id;
        
    }

    /**
     * Method to check in an item.
     * @param   integer $id The id of the row to check out.
     * @return  boolean True on success, false on failure.
     * @since    1.6
     */
    public function checkin($id = null)
    {
        // Get the id.
        $id = (!empty($id)) ? $id : (int) $this->getState('picture.id');
        
        if ($id) {
            // Initialise the table
            $table = $this->getTable();

            // Attempt to check the row in.
            if (method_exists($table, 'checkin')) {
                if (!$table->checkin($id)) {
                    return false;
                }
            }
        }

        return true;
        
    }

    /**
     * Method to check out an item for editing.
     * @param   integer $id The id of the row to check out.
     * @return  boolean True on success, false on failure.
     * @since    1.6
     */
    public function checkout($id = null)
    {
        // Get the user id.
        $id = (!empty($id)) ? $id : (int) $this->getState('picture.id');
        
        if ($id) {
            // Initialise the table
            $table = $this->getTable();

            // Get the current user object.
            $user = GagalleryHelper::getSpecificUser();

            // Attempt to check the row out.
            if (method_exists($table, 'checkout')) {
                if (!$table->checkout($user->id, $id)) {
                    return false;
                }
            }
        }

        return true;
        
    }

    /**
     * Method to get the form.
     * The base form is loaded from XML
     * @param   array   $data     An optional array of data for the form to interogate.
     * @param   boolean $loadData True if the form is to load its own data (default case), false if not.
     * @return  Form    A Form object on success, false on failure
     * @since   1.6
     */
    public function getForm($data = array(), $loadData = true)
    {
        // Get the form.
        $form = $this->loadForm('com_gagallery.picture', 'pictureform', array(
                    'control'   => 'jform',
                    'load_data' => $loadData ) );

        if (empty($form)) {
                return false;
        }

        return $form;
    }

    /**
     * Method to get the data that should be injected in the form.
     * @return    array  The default data is an empty array.
     * @since    1.6
     */
    protected function loadFormData()
    {
        $data = Factory::getApplication()->getUserState('com_gagallery.edit.picture.data', array());

        if (empty($data)) {
            $data = $this->getItem();
        }

        if ($data) {

            return $data;
        }

        return array();
    }

    /**
     * Method to save the form data.
     * @param   array $data The form data
     * @return bool
     * @throws Exception
     * @since 1.6
     */
    public function save($data)
    {
        $id    = (!empty($data['id'])) ? $data['id'] : (int) $this->getState('picture.id');
        $state = (!empty($data['state'])) ? 1 : 0;
        $data['exp_date'] = !empty($data['exp_date']) ?: null;
        $user  = GagalleryHelper::getSpecificUser();

        $params = ComponentHelper::getParams('com_gagallery');
        $fileLocation  = $params->get( 'file_loc' );
        $fileLocation  = (!empty($data['loadfolder'])) ? $data['loadfolder'] : $fileLocation;
        // img_name is an array of the file details
        $data['img_name'] = (!empty($data['img_name']) && is_array($data['img_name'])) ? 'images/'.$fileLocation.'/'.$data['img_name']['name'] : $data['img_name'];

        if ($id) {
            // Check the user can edit this item
            $authorised = $user->authorise('core.edit', 'com_gagallery') || $authorised = $user->authorise('core.edit.own', 'com_gagallery');
        } else {
            // Check the user can create new items in this section
            $authorised = $user->authorise('core.create', 'com_gagallery');
        }

        if ($authorised !== true) {
            throw new \Exception(Text::_('JERROR_ALERTNOAUTHOR'), 403);
        }

        $table = $this->getTable();

        if ($table->save($data) === true) {
            return $table->id;
        } else {
            return false;
        }
        
    }

    /**
     * Method to delete data
     * @param   int $pk Item primary key
     * @return  int  The id of the deleted item
     * @throws Exception
     * @since 1.6
     */
    public function delete($pk)
    {
        $user = GagalleryHelper::getSpecificUser();

        if (empty($pk)) {
            $pk = (int) $this->getState('picture.id');
        }

        if ($pk == 0 || $this->getItem($pk) == null) {
            throw new \Exception(Text::_('COM_GAGALLERY_ITEM_DOESNT_EXIST'), 404);
        }

        if ($user->authorise('core.delete', 'com_gagallery') !== true) {
            throw new \Exception(Text::_('JERROR_ALERTNOAUTHOR'), 403);
        }

        $table = $this->getTable();

        if ($table->delete($pk) !== true) {
            throw new \Exception(Text::_('JERROR_FAILED'), 501);
        }

        return $pk;
        
    }

    /**
     * Check if data can be saved
     * @return bool
     */
    public function getCanSave()
    {
        $table = $this->getTable();

        return $table !== false;
    }
    
    public function uplattachfile($data, $folder = 0)
	{
        if(GagalleryHelper::getSpecificUser()->authorise('core.attupload', 'com_gagallery') !== true){
            throw new \Exception(Text::_('JERROR_ALERTNOAUTHOR'),403);
            return false;
        }

        $params = ComponentHelper::getParams('com_gagallery');
        $file_loc  = $params->get( 'file_loc' );
        if (!$folder) { $folder = $file_loc;  }
        $safeFileOptions  = $params->get( 'safe_files' );
        $file_ext = strtolower(substr($data['img_name']['name'],-3));

		if (!in_array($file_ext, $safeFileOptions)) {
			Factory::getApplication()->enqueueMessage(Text::sprintf('COM_GAGALLERY_IMGTYPE_NOTALLOWED',$file_ext, $data['img_name']), 'warning');
			return false;
		}

        if (file_exists('file://'.$data['img_name']['tmp_name'])) {
			$fileName = File::makeSafe($data['img_name']['name']);
			$fileName = str_replace(' ', '_', $fileName);
			$src = $data['img_name']['tmp_name'];

			$path = Path::clean( JPATH_SITE . '/images/'.$folder );
			$destfile = $path.'/'.$fileName;

			if ( File::upload($src, $destfile, false, false, $safeFileOptions) ) {
				Factory::getApplication()->enqueueMessage(Text::sprintf('COM_GAGALLERY_IMGTYPE_UPLOADED',$fileName), 'message');
			} else {
				Factory::getApplication()->enqueueMessage(Text::sprintf('COM_GAGALLERY_IMGTYPE_NOTUPLOADED',$fileName), 'danger');
			}

			return 'images/'.$folder.'/'.$fileName;
  		} else {
			return 'no title';
		}

    }
    
    public function autoLoad($data)
	{
        $user = GagalleryHelper::getSpecificUser();
        $date = GagalleryHelper::getTodaysDate();
        $today = date_format($date,'Y-m-d H:i:s');

        if($user->authorise('core.attupload', 'com_gagallery') !== true){
            throw new \Exception(Text::_('JERROR_ALERTNOAUTHOR'),403);
            return false;
        }

        $app = Factory::getApplication();

        $params = ComponentHelper::getParams('com_gagallery');
        $safeFileOptions  = $params->get( 'safe_files' );
        $file_loc  = $params->get( 'file_loc' );

        if (is_dir('images/'.$file_loc.'/'.$data['loadfolder'])) {

            // get all the files from selected folder
            $files = Folder::files('images/'.$file_loc.'/'.$data['loadfolder'], '.', false, false, $safeFileOptions, array('^\..*', '.*~'), false);

            foreach ($files as $file) {
    
        		$file_ext = File::getExt($file);
    
        		if (!in_array($file_ext, $safeFileOptions)) {
        			$app->enqueueMessage(Text::sprintf('COM_GAGALLERY_IMGTYPE_NOTALLOWED',$file_ext, $file), 'warning');
        			continue;
        		}
        		
        		// create a new record for the gallery
        		$fileName = File::stripExt($file);
        		$newRec = new \stdClass();
        		$newRec->id = 0;
        		$newRec->state = 1;
        		$newRec->created_date = $today;
        		$newRec->created_by = $user->id;
        		$newRec->title = str_replace('_', ' ', $fileName);
        		$newRec->cat_id = $data['cat_id'];
        		$newRec->img_name = 'images/'.$file_loc.'/'.$data['loadfolder'].'/'.$file;
        		$newRec->comment = 'Auot loaded';
        		
                try {
                    $result = Factory::getDbo()->insertObject('#__gagallery_pictures', $newRec, 'id');
                    $app->enqueueMessage(Text::sprintf('COM_GAGALLERY_IMG_LOADED',$fileName), 'success');
                } catch (RuntimeException $e) {
                    $app->enqueueMessage(Text::sprintf('COM_GAGALLERY_SAVE_FAILED',$fileName), 'warning');
                }

        	}
            return true;
        } else {
            return false;
        }

    }
}
